<?php 
/*
Web based document management system
Copyright (C) 2003,2004 Hendri Hondorp, Dennis Reidsma, Arthur van Bunningen, Wietse Balkema

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/


/*
This file offers several functions to display publications in Aigaion
Each function returns a string that contains (parts of) a display.

  getPublicationList($type, $option = "", $checkbox = "true")
    returns a list of publications, sorted by $type
    allowed types are:
    - year
    - title
    - topic -> when $ID is passed via command line, only topics of $ID
               otherwise same as year. $option can also be used
    - author
    - journal
    - unassigned
    - newest -> when $ID is passed via command line, only pubs newer than $ID
                are shown, otherwise the 10 newest pubs. $option can also be used
    $option can be used for passing options like year, author, topic for use
    without commandline.

  getPublicationByID($pub_id)
    returns a detailed overview of a single publication.

  getPublicationSummaryByID($pub_id, $tableFormat = "inTable", $displayFirst = "author", $checkbox = "false") 
    returns a summary of a publication.
    allowed tableFormat:
    - inTable for displaying in a table 1st cell: attachments etc, 2nd cell: actual summary
    - plain   for displaying without a table
    allowed displayFirst:
    - author for beginning the summary with the first authorname
    - title  for beginning the summary with the title
    
  getPublicationSummaryByRow($publicationRow, $tableFormat = "inTable", $displayFirst = "author", $checkbox = "false") 
    returns a summary of a publication.
    allowed tableFormat:
    - inTable for displaying in a table 1st cell: attachments etc, 2nd cell: actual summary
    - plain   for displaying without a table
    allowed displayFirst:
    - author for beginning the summary with the first authorname
    - title  for beginning the summary with the title

  getPublicationSummaryByRowNoNotes($publicationRow, $tableFormat = "inTable", $displayFirst = "author", $checkbox = "false") 
    returns a summary of a publication, without the notes.
    allowed tableFormat:
    - inTable for displaying in a table 1st cell: attachments etc, 2nd cell: actual summary
    - plain   for displaying without a table
    allowed displayFirst:
    - author for beginning the summary with the first authorname
    - title  for beginning the summary with the title

  getLinkToPublicationDisplayPage($pub_id, $title)
    returns a link to the complete display of the publication.
    
  getLinkToPublicationEditPage($pub_id, $size = "normal")
    returns an icon as link to the publication edit page.
    $size specifies the size of the icon to use.
    allowed size:
    - normal
    - small
    
  getLinkToPublicationDeletePage($pub_id, $size = "normal")
    returns an icon as link to the publication edit page.
    $size specifies the size of the icon to use.
    allowed size:
    - normal
    - small
    
  getTypeSpecificElements($publicationRow)
    returns all type specific elements of a publication.
    
  getTypeSpecificElementsSummary($publication)
    returns all type specific elements for a summary.
    
  getElement($elements, $element, $data, $name, $tableFormat = "inTable")
    returns the element $data when $element is present in the array $elements.
    $tabbleFormat us used to differentiate between normal listing and summary
    listing. Summary uses "plain".
    allowed tableFormat:
    - inTable
    - plain
  
*/

require_once($DIR."checkBase.php"); //if this script is not called from within one of the base pages, redirect to frontpage
include_once($DIR."lib/functions/publicationfunctions.php");
include_once($DIR."lib/functions/bibtexfunctions.php");
include_once($DIR."lib/functions/authorfunctions.php");
include_once($DIR."lib/functions/specialcharfunctions.php");
include_once($DIR."lib/displays/attachmentdisplays.php");
include_once($DIR."lib/displays/notedisplays.php");
include_once($DIR."lib/displays/exportdisplays.php");
include_once($DIR."lib/displays/authordisplays.php");
include_once($DIR."lib/editforms/multipublicationeditforms.php");
include_once($DIR."javascript/collectcheckedpublications.js");


function getPublicationList($type, $option = "", $checkbox = "true") 
{
  //allowed types: year, topic, journal
  //requires: $ID (topic_id) on $type = topic
  
  if ($type == "")
  {
    return "<div class='errormessage'>An error occurred while fetching the type of publication listing. Please try again or ask your Aigaion administrator.</div>\n";
  }
  
  $nPubsPerPage = $_SESSION["liststyle"];
  if (isset($_REQUEST["pageNr"]))
  {
    $pageNr = $_REQUEST["pageNr"] - 1;
  }
  else
  {
    $pageNr = 0;
  }
  
  $result = "";
  $tableFormat = "inTable";
  $displayFirst = $_SESSION["summarystyle"];
  
  //create page selection table if necessary
  $pageTable = "";
  if ($nPubsPerPage != 0)
  {
	  if ($type != "topic")
	  {
	    $Q = mysql_query("SELECT COUNT(*) FROM publication");
	    $R = mysql_fetch_array($Q);
	    $nPubs = $R["COUNT(*)"];
	  }
	  else
	  {
	    $topic_id = "";
	    if ($option == "")
	    {
	      if (isset($_REQUEST["ID"])) {
	        $topic_id = $_REQUEST["ID"];
	      }
	    }
	    else
	    {
	      $topic_id = $option;
	    }
	    $Q = mysql_query("SELECT COUNT(*) FROM publication, topicpublication WHERE publication.pub_id = topicpublication.pub_id AND topicpublication.topic_id = ".$topic_id);
	    $R = mysql_fetch_array($Q);
	    $nPubs = $R["COUNT(*)"];
	  }
	  if ($nPubs > $nPubsPerPage) 
	  {
	    $pageTable  = "<table width=100%>";
	    $pageTable .= "<tr><td align=center>";
	    $count = 0;
	    $start = 1;
	    $end   = $nPubsPerPage;
	    $pageTable .= "|";
	    while ($count * $nPubsPerPage < $nPubs)
	    {
	      $start = ($count*$nPubsPerPage)+1;
	      $end   = ($count+1)*$nPubsPerPage;
	      if ($end > $nPubs)
	      {
	        $end = $nPubs;
	      }
	      
	      if ($pageNr != $count)
	      {
	        if ($type != "topic")
	        {
	          $pageTable .= ' <a href="index.php?page=publication&kind=list&type='.$type.'&pageNr='.($count+1).'">'.$start.'..'.$end.'</a> |';;
	        }
	        else
	        {
	          $pageTable .= ' <a href="index.php?page=topic&topic_id='.$topic_id.'&pageNr='.($count+1).'">'.$start.'..'.$end.'</a> |';
	        }  
	        
	      }
	      else
	      {
	        $pageTable .= "<b>".$start."..".$end."</b>|";
	      }
	      $count++;
	    }
	    $pageTable .= "</td></tr></table>\n";
	  }
  }
  //end of page selection table
  
  
  switch ($type){
    case "year":
      //display all publications in the DB, sorted on year, descending.
      $limit = "";
      if ($_SESSION["liststyle"] != "0")
      	$limit = " LIMIT ".$pageNr*$nPubsPerPage.", ".$nPubsPerPage;
      
      if ($displayFirst == "author")
      {
        $Q  = mysql_query("SELECT DISTINCT publication.* FROM publication LEFT JOIN publicationauthor ON publication.pub_id = publicationauthor.pub_id, author WHERE ((publication.pub_id = publicationauthor.pub_id AND publicationauthor.author = author.ID AND publicationauthor.rank=1) OR (publicationauthor.author IS NULL)) GROUP BY publication.pub_id ORDER BY year DESC, is_editor DESC, surname, title".$limit);
      }
      else //displayfirst == "title"
      {
        $Q = mysql_query("SELECT * FROM publication ORDER BY year DESC, title".$limit);
      }
      $result .= "<p class=header>All publications in the database, sorted on year</p>\n";

      if ($checkbox == "true")
      {
        $result .= getAddCheckedToSelectedTopicForm();
        $result .= "<table><tr><td>Export selected publications to:</td><td>".getExportSelectedForm("BibTeX")."</td>";
        $result .= "<td>".getExportSelectedForm("RIS")."</td></tr></table>";
      }
        
      $result .= $pageTable;  

      $result .= "<table class=vlist>";
      
      $year = "";
      while ($R = mysql_fetch_array($Q))
      {
        if ($year != $R['year'])
        {
        	//display header
          if ($R['year'] != "0000")
          {	$year = $R['year'];
          	$result .= "<td colspan=2><p class=header>".$year."</td></tr>\n";
          }
          else if ($year != "Unknown year")
          {	
          	$year = "Unknown year";
          	$result .= "<td colspan=2><p class=header>".$year."</td></tr>\n";
          }
        }
        $result .= getPublicationSummaryByRow($R, $tableFormat, $displayFirst, $checkbox);
      }
      $result .= "</table>";
      $result .= $pageTable;  
      break;

    case "title":
      //display all publications in the DB, sorted on title
      $limit = "";
      if ($_SESSION["liststyle"] != "0")
      	$limit = " LIMIT ".$pageNr*$nPubsPerPage.", ".$nPubsPerPage;
      
      $Q = mysql_query("SELECT * FROM publication ORDER BY title".$limit);
      $result .= "<p class=header>All publications in the database, sorted on title</p>\n";

      if ($checkbox == "true")
      {
        $result .= getAddCheckedToSelectedTopicForm();
        $result .= "<table><tr><td>Export selected publications to:</td><td>".getExportSelectedForm("BibTeX")."</td>";
        $result .= "<td>".getExportSelectedForm("RIS")."</td></tr></table>";
      }
      
      $result .= $pageTable;  
      
      $result .= "<table class=vlist>";
      $key = "";
      
      while ($R = mysql_fetch_array($Q))
      {
        if ($key != strtoupper($R['title']{0}))
        {
          $key = strtoupper($R['title']{0});
          //display header
          $result .= "<td colspan=2><p class=header>".$key."</td></tr>\n";
        }
        $result .= getPublicationSummaryByRow($R, $tableFormat, $displayFirst, $checkbox);
      }
      $result .= "</table>";
      $result .= $pageTable;  
      break;

    case "topic":
      //display all publications of the selected topic, if no topic is set, display all topics sorted by year
      $topic_id = "";
      if ($option == "")
      {
        if (isset($_REQUEST["ID"])) {
          $topic_id = $_REQUEST["ID"];
        }
      }
      else
      {
        $topic_id = $option;
      }
      
      $limit = "";
      if ($_SESSION["liststyle"] != "0")
      	$limit = " LIMIT ".$pageNr*$nPubsPerPage.", ".$nPubsPerPage;
      
      
      $result .= "<table class=vlist>";
      $year = "";
      
      //when no topic_id, set to top
      if ($topic_id == "")
      {
        $topic_id = 1;
        $result .= "<p class=header>All publications in the database</p>\n";
      }
      else
      {
        $Q = mysql_query("SELECT name from topic where ID=".$topic_id.$limit);
        //echo mysql_error();
        //echo "SELECT name from topic where ID=".$topic_id.$limit;
        $R = mysql_fetch_array($Q);
        $result .= "<p class=header>Publications about ".$R['name']."</p>";
      }

      if ($checkbox == "true")
      {
        $result .= getAddCheckedToSelectedTopicForm();
        $result .= getRemoveCheckedFromTopicForm($topic_id);
        $result .= "<br><table><tr><td valign=center>Export selected publications to:</td><td>".getExportSelectedForm("BibTeX")."</td>";
        $result .= "<td>".getExportSelectedForm("RIS")."</td></tr></table>";
      }
      
      $result .= $pageTable;
      $result .= "<table class=vlist>";
      if ($displayFirst == "author")
      {
        $Q = mysql_query("SELECT DISTINCT publication.* FROM publication LEFT JOIN publicationauthor ON publication.pub_id = publicationauthor.pub_id, author, topicpublication WHERE publication.pub_id = topicpublication.pub_id AND topicpublication.topic_id = ".$topic_id." AND ((publication.pub_id = publicationauthor.pub_id AND publicationauthor.author = author.ID AND publicationauthor.rank=1) OR (publicationauthor.author IS NULL)) GROUP BY publication.pub_id ORDER BY year DESC, is_editor DESC, surname, title".$limit);
          echo mysql_error();

      }
      else //displayfirst == "title"
      {
        $Q = mysql_query("SELECT publication.* FROM publication, topicpublication WHERE publication.pub_id = topicpublication.pub_id AND topicpublication.topic_id = ".$topic_id." ORDER BY year DESC, title".$limit);
      }

      $year = "";
      while ($R = mysql_fetch_array($Q))
      {
        if ($year != $R['year'])
        {
        	//display header
          if ($R['year'] != "0000")
          {	$year = $R['year'];
          	$result .= "<td colspan=2><p class=header>".$year."</td></tr>\n";
          }
          else if ($year != "Unknown year")
          {	
          	$year = "Unknown year";
          	$result .= "<td colspan=2><p class=header>".$year."</td></tr>\n";
          }
        }
				$result .= getPublicationSummaryByRow($R, $tableFormat, $displayFirst, $checkbox);
      }
      $result .= "</table>";
      $result .= $pageTable;  

      break;

    case "author":
      //display all publications that are assigned to a single author
      $author="";
      if ($option == "")
      {
        if (isset($_REQUEST["author"])) {
          $author=$_REQUEST["author"];
        }
      }
      else
      {
        $author = $option;
      }
      
      //as author
      $Q = mysql_query("SELECT DISTINCT publication.* FROM publication, publicationauthor WHERE author=".$author." AND is_editor='N' and publication.pub_id=publicationauthor.pub_id GROUP BY publication.pub_id ORDER BY year DESC, publicationauthor.rank DESC");
      if (mysql_num_rows($Q)) 
      {
        $result .= "<p class=header>Publications as Author</p>\n";

        if ($checkbox == "true")
	      {
	        $result .= getAddCheckedToSelectedTopicForm();
	        $result .= "<table><tr><td>Export selected publications to:</td><td>".getExportSelectedForm("BibTeX")."</td>";
	        $result .= "<td>".getExportSelectedForm("RIS")."</td></tr></table>";
	      }
      
        $result .= "<table class=vlist>";

        $year = "";    
        while ($R = mysql_fetch_array($Q))
        {
          if ($year != $R['year'])
	        {
	        	//display header
	          if ($R['year'] != "0000")
	          {	$year = $R['year'];
	          	$result .= "<td colspan=2><p class=header>".$year."</td></tr>\n";
	          }
	          else if ($year != "Unknown year")
	          {	
	          	$year = "Unknown year";
	          	$result .= "<td colspan=2><p class=header>".$year."</td></tr>\n";
	          }
	        }
          $result .= getPublicationSummaryByRow($R, $tableFormat, $displayFirst, $checkbox);
        }
        $result .= "</table>";
      }
      
      //as editor
      $Q = mysql_query("SELECT DISTINCT publication.* FROM publication, publicationauthor WHERE author=".$author." AND is_editor='Y' and publication.pub_id=publicationauthor.pub_id GROUP BY publication.pub_id ORDER BY year DESC, publicationauthor.rank DESC");
      if (mysql_num_rows($Q)) 
      {
        $result .= "<p class=header>Publications as Editor</p>\n";
        $result .= "<table class=vlist>";

        $year = "";    
        while ($R = mysql_fetch_array($Q))
        {
          if ($year != $R['year'])
	        {
	        	//display header
	          if ($R['year'] != "0000")
	          {	$year = $R['year'];
	          	$result .= "<td colspan=2><p class=header>".$year."</td></tr>\n";
	          }
	          else if ($year != "Unknown year")
	          {	
	          	$year = "Unknown year";
	          	$result .= "<td colspan=2><p class=header>".$year."</td></tr>\n";
	          }
	        }
          $result .= getPublicationSummaryByRow($R, $tableFormat, $displayFirst, $checkbox);
        }
        $result .= "</table>";
      }
  
      break;

    case "journal":
      $journal = "";
      
      $limit = "";
      if ($_SESSION["liststyle"] != "0")
      	$limit = " LIMIT ".$pageNr*$nPubsPerPage.", ".$nPubsPerPage;
      
      if ($displayFirst == "author")
      {
        //first, get articles that are assigned to a journal
        $Q = mysql_query("SELECT DISTINCT publication.* FROM publication LEFT JOIN publicationauthor ON publication.pub_id = publicationauthor.pub_id, author WHERE ((publication.pub_id = publicationauthor.pub_id AND publicationauthor.author = author.ID AND publicationauthor.rank=1) OR (publicationauthor.pub_id IS NULL)) AND journal != '' GROUP BY publication.pub_id ORDER BY journal, year DESC, is_editor DESC, surname, title".$limit);

        if ((mysql_num_rows($Q) < $nPubsPerPage) || ($_SESSION["liststyle"] == "0"))
        {
        	if ($_SESSION["liststyle"] == "0")
        		$limit = "";
        	else
        	{
	          //then get articles without journal
	          $nRemain = $nPubsPerPage - mysql_num_rows($Q);
	         
	          $Q3 = mysql_query("SELECT COUNT(*) FROM publication WHERE journal != ''");
	          $R = mysql_fetch_array($Q3);
	          $nJournPubs = $R["COUNT(*)"];
	          $lastJournalPage = ceil($nJournPubs/$nPubsPerPage);
	          if (($pageNr+1) != $lastJournalPage) //others are not relevant, since we do not arrive here on normal journal pages
	
	          { 
	            $start = (($pageNr) - $lastJournalPage)*$nPubsPerPage;
	            $start+= ($lastJournalPage*$nPubsPerPage) - $nJournPubs;
	          }
	          else
	          {
	            $start = 0;
	          }
	          $limit = " LIMIT ".$start.", ".$nRemain;
	        }
         
          $Q2 = mysql_query("SELECT DISTINCT publication.* FROM publication LEFT JOIN publicationauthor ON publication.pub_id = publicationauthor.pub_id, author WHERE ((publication.pub_id = publicationauthor.pub_id AND publicationauthor.author = author.ID AND publicationauthor.rank=1) OR (publicationauthor.pub_id IS NULL)) AND journal = '' GROUP BY publication.pub_id ORDER BY year DESC, is_editor DESC, surname, title".$limit);

        }
      }
      else //displayfirst == "title"
      {
        //first, get articles that are assigned to a journal
        $Q = mysql_query("SELECT * FROM publication WHERE journal != '' ORDER BY journal, year DESC, title LIMIT ".$pageNr*$nPubsPerPage.", ".$nPubsPerPage);

        if ((mysql_num_rows($Q) < $nPubsPerPage) || ($_SESSION["liststyle"] == "0"))
        {
          if ($_SESSION["liststyle"] == "0")
        		$limit = "";
        	else
        	{
						//then get articles without journal
	          $nRemain = $nPubsPerPage - mysql_num_rows($Q);
	         
	          $Q3 = mysql_query("SELECT COUNT(*) FROM publication WHERE journal != ''");
	          $R = mysql_fetch_array($Q3);
	          $nJournPubs = $R["COUNT(*)"];
	          if (($pageNr+1) != $lastJournalPage) //others are not relevant, since we do not arrive here on normal journal pages
	
	          { 
	            $start = (($pageNr) - $lastJournalPage)*$nPubsPerPage;
	            $start+= ($lastJournalPage*$nPubsPerPage) - $nJournPubs;
	          }
	          else
	          {
	            $start = 0;
	          }
 	          $limit = " LIMIT ".$start.", ".$nRemain;
	        }

          $Q2 = mysql_query("SELECT * FROM publication WHERE journal = '' ORDER BY journal, year DESC, title".$limit);

        }
      }
     
     
      $result .= "<p class=header>All publications in the database, sorted on journal</p>\n";

      if ($checkbox == "true")
      {
        $result .= getAddCheckedToSelectedTopicForm();
        $result .= "<table><tr><td>Export selected publications to:</td><td>".getExportSelectedForm("BibTeX")."</td>";
        $result .= "<td>".getExportSelectedForm("RIS")."</td></tr></table>";
      }
      
      $result .= $pageTable; 
      $result .= "<table class=vlist>";
      $year = "";
     
      while ($R = mysql_fetch_array($Q))
      {
        if ($journal != $R['journal'])
        {
          $journal = $R['journal'];
          //display header
          $result .= "<td colspan=2><p class=header>".$journal."</td></tr>\n";
        }
        $result .= getPublicationSummaryByRow($R, $tableFormat, $displayFirst, $checkbox);
      }
      if (mysql_num_rows($Q2) > 0)
      {
        //header: no assigned journal
        $result .= "<td colspan=2><p class=header>Other publications in the database, sorted on year</td></tr>\n";
      }
      $year = "";
      while ($R = mysql_fetch_array($Q2))
      {
        if ($year != $R['year'])
        {
        	//display header
          if ($R['year'] != "0000")
          {	$year = $R['year'];
          	$result .= "<td colspan=2><p class=header>".$year."</td></tr>\n";
          }
          else if ($year != "Unknown year")
          {	
          	$year = "Unknown year";
          	$result .= "<td colspan=2><p class=header>".$year."</td></tr>\n";
          }
        }
        $result .= getPublicationSummaryByRow($R, $tableFormat, $displayFirst, $checkbox);
      }
      $result .= "</table>"; 
      $result .= $pageTable; 
     
      break; 
      
    case "unassigned":
      //display all publications in the DB that are not assigned to any topic
      $limit = "";
      if ($_SESSION["liststyle"] != "0")
      	$limit = " LIMIT ".$pageNr*$nPubsPerPage.", ".$nPubsPerPage;
			
			if ($displayFirst == "author")
      {
        $Q = mysql_query("SELECT DISTINCT publication.* FROM publication LEFT JOIN publicationauthor ON publication.pub_id = publicationauthor.pub_id, author WHERE ((publication.pub_id = publicationauthor.pub_id AND publicationauthor.author = author.ID AND publicationauthor.rank=1) OR (publicationauthor.author IS NULL)) GROUP BY publication.pub_id ORDER BY year DESC, is_editor DESC, surname, title".$limit);
      }
      else //displayfirst == "title"
      {
        $Q = mysql_query("SELECT * FROM publication ORDER BY year DESC, title".$limit);
      }

      $result .= "<p class=header>Publications that are not assigned to any topic</p>\n";

      if ($checkbox == "true")
        $result .= getAddCheckedToSelectedTopicForm();

			$result .= $pageTable;
      $result .= "<table class=vlist>";
      
      while ($R = mysql_fetch_array($Q)) 
	    {
		    $Q2 = mysql_query("SELECT COUNT(*) FROM topicpublication WHERE (topic_id!=1) AND (pub_id=".$R['pub_id'].")");
		    $R2 = mysql_fetch_array($Q2);
		    if ($R2["COUNT(*)"] == 0) 
		    {
          $result .= getPublicationSummaryByRow($R, $tableFormat, $displayFirst, $checkbox);
		    }
	    }

			$result .= $pageTable;
      $result .= "</table>";
      
      break;
    
    case "newest":
      //display the 10 newest publications or the publications newer than $ID
      $oldest_id = "";
      if ($option == "")
      {
        if (isset($_REQUEST["ID"])) {
          $oldest_id = $_REQUEST["ID"];
        }
      }
      else
      {
        $oldest_id = $option;
      }
      
      if ($oldest_id != "")
      {
        $Q = mysql_query("SELECT * FROM publication WHERE pub_id >= ".$oldest_id." ORDER BY pub_id DESC");
      }
      else
      {
        $Q = mysql_query("SELECT * FROM publication ORDER BY pub_id DESC LIMIT 10");
      }
      
      $result .= "<p class=header>Newly added publications</p>\n";

      if ($checkbox == "true")
      {
        $result .= getAddCheckedToSelectedTopicForm();
        $result .= "<table><tr><td>Export selected publications to:</td><td>".getExportSelectedForm("BibTeX")."</td>";
        $result .= "<td>".getExportSelectedForm("RIS")."</td></tr></table>";
      }
      
      $result .= "<table class=vlist>";
      
      while ($R = mysql_fetch_array($Q)) 
	    {
        $result .= getPublicationSummaryByRow($R, $tableFormat, $displayFirst, $checkbox);
	    }

      $result .= "</table>";
      
      break;
    default:
      break;
    
  }
  
  return $result;
}
  

function getPublicationByID($pub_id) {
  global $URL, $DIR;
  include_once($DIR."treefunctions.php");
  include_once($DIR."lib/editforms/publicationeditforms.php");
  
  if ($pub_id == "")
  {
    return "<div class='errormessage'>An error occurred while fetching the publication ID. Please try again or ask your Aigaion administrator.</div>\n";
  }

  $result = "";

  $result .= "<p class=header>Publication &nbsp;&nbsp;&nbsp;";
  
  if (haveRights("EDIT")) {
    $result .= getLinkToPublicationEditPage($pub_id)." ".getLinkToPublicationDeletePage($pub_id,"normal",$URL);
  }
  $result .= "</p>\n";

  $result .= "<table class=vlist cellspacing='0' width='100%'>";

  //get the publication data
  $Q = mysql_query("SELECT * FROM publication WHERE pub_id =".$pub_id.";");
  if(!$Q)
  {
    return "<div class='errormessage'>The requested publication was not found</div>";
  }
  
  $publication = mysql_fetch_array($Q);
  if(!$publication)
  {
    return "<div class='errormessage'>The requested publication was not found</div>";
  }
  //check for special chars
  if ($publication['specialchars'] == "TRUE")
  {
    prettyPrintBibCharsFromArray($publication);
  }
 
  //first, display required fields for all publication types
  $result .= "<tr><td width=25%>Type of publication:</td><td>".$publication["type"]."</td></tr>\n";
  
  $Q = mysql_query("SELECT * FROM person WHERE ID = ".$publication['entered_by']);
  $R = mysql_fetch_array($Q);
  $result .= "<tr><td>Entered by:</td><td>".$R['abbreviation']."</td></tr>\n";

  if ($publication['title']) {
    $title = $publication['title'];
    
    quotesToHTMLFromString($title);
    $result .= "<tr><td>Title</td><td>";
    
    $Q = mysql_query("SELECT * FROM publicationfile WHERE pub_id =".$pub_id." AND ismain='TRUE'");
    if ($Q && ($R = mysql_fetch_array($Q)) && haveRights("PDF")) {
      $result .= getAttachmentDownloadLink($R, $title); 
    }
    else {
      $result .= $publication['title'];
    }
    $result .= "</td></tr>\n";
  }
  
  $result .= "<tr><td>Bibtex cite ID</td><td>".$publication['bibtex_id']."</td></tr>\n";
  
  //get publicationtype specific fields
  $result .= getTypeSpecificElements($publication);
  
  //get authors
  $Q = mysql_query("SELECT * FROM publicationauthor WHERE pub_id = ".$pub_id." AND is_editor='N' ORDER by rank;");
  if (mysql_num_rows($Q) > 0) {
    $result .= "<tr><td valign=top>Authors</td><td><table class=vlist cellspacing='0'>";
    
    while ($R = mysql_fetch_array($Q)) {
      $Q2 = mysql_query("SELECT * FROM author WHERE author.ID=".$R["author"]);
      $R2 = mysql_fetch_array($Q2);
      
      $result .= "<tr><td>".getLinkToAuthorPageFromRow($R2)."</a></td></tr>\n";
    }
    $result .= "</table></td></tr>\n";
  }

  //get editors
  $Q = mysql_query("SELECT * FROM publicationauthor WHERE pub_id = ".$pub_id." AND is_editor='Y' ORDER by rank;");
  if (mysql_num_rows($Q) > 0) {
    $result .= "<tr><td valign=top>Editors</td><td><table class=vlist cellspacing='0'>";
    
    while ($R = mysql_fetch_array($Q)) {
      $Q2 = mysql_query("SELECT * FROM author WHERE author.ID=".$R["author"]);
      $R2 = mysql_fetch_array($Q2);
      
      $result .= "<tr><td>".getLinkToAuthorPageFromRow($R2)."</a></td></tr>\n";
    }
    $result .= "</table></td></tr>\n";
  }
  
  //subscribed topics
  $Q = mysql_query("SELECT * FROM topicpublication, persontopic WHERE pub_id=".$pub_id." AND persontopic.person_id=".$_SESSION["personID"]." AND persontopic.topic_id=topicpublication.topic_id");
  if ($Q) {
    while($R = mysql_fetch_array($Q)) {
      $subscribedTopics[] = $R['topic_id'];
    }
  }
  
  //all topics
  $Q = mysql_query("SELECT * FROM topicpublication WHERE pub_id=".$pub_id);
  if ($Q) {
    while($R = mysql_fetch_array($Q)) {
      $allTopics[] = $R['topic_id'];
    }
  }
  
  //find out if there are other classifications
  $otherTopics = array_diff( array_merge($subscribedTopics, $allTopics), $subscribedTopics);
  if (count($otherTopics) > 0) {
    $bOtherClassifications = true;
  }
  else {
    $bOtherClassifications = false;
  }
  
  //make subscribed topic string and get the topicIDs in an array
  $subscribedTopicsString = "";
  $subscribedTopicIDs = array();
  if (count($subscribedTopics) > 0) {
    foreach ($subscribedTopics as $key => $topicID) {
      if ($subscribedTopicsString != ""){
        $subscribedTopicsString.=",";
      }
      $subscribedTopicsString .= $topicID;
      $subscribedTopicIDs[] = $topicID;
    }
  }
  
  
  //and get the other classification IDs in an array
  if (bOtherClassifications) {
    $otherTopicIDs = array();
    foreach ($otherTopics as $key => $topicID) {
      $otherTopicIDs[] = $topicID;
    } 
  }
  
  //make the actual topics output
  $result .= "<tr><td valign=top>Topics <br/><a href class='islink' onclick=\"javascript:window.open('indexlight.php?page=listsubjectsandstore&pub_id=".$pub_id."&topics=".$subscribedTopicsString."','open_window','resizable, scrollbars,width=850, height=800, dependent, left=0, top=0');\">(edit)</a></td><td valign=top>";
  
  if ($bOtherClassifications) {
    $result .= "<span class=\"expanded\" id=\"showother\" onclick=\"javascript:document.getElementById('subscr').className='collapsed';document.getElementById('notsubscr').className='expanded';document.getElementById('showown').className='expanded';document.getElementById('showother').className='collapsed';\">=<span class=islink >SEE CLASSIFICATION DIFFERENCE FROM OTHERS</span>=</span>";
    $result .= "<span class=\"collapsed\" id=\"showown\" onclick=\"javascript:document.getElementById('subscr').className='expanded';document.getElementById('notsubscr').className='collapsed';document.getElementById('showother').className='expanded';document.getElementById('showown').className='collapsed';\">=<span class=islink >SEE OWN CLASSIFICATION</span>=</span><br/>";
  }
  
  $result .= "<div id='subscr' class='expanded'>";
  
  if (count($subscribedTopics) > 0) {
    $result .= getTopicSimpleTree(makeTopicTree($subscribedTopicIDs));
  }
  $result .= "</div>";
  
  if ($bOtherClassifications)
  {
    $result .= "<div id='notsubscr' class=collapsed>";
    $result .= getTopicSimpleTree(makeTopicTree($otherTopicIDs));
    $result .= "</div>";
  }
  $result .= "</td></tr>\n";
  
  //bibtex output link
  $result .= "<tr><td>BibTeX</td><td>".getBibtexLinkByID($pub_id, "text")."</td></tr>\n";
  $result .= "<tr><td>RIS</td><td>".getRISLinkByID($pub_id, "text")."</td></tr>\n";
  
  if (trim($publication['crossref']) != "")
  {
    $result .= "<tr><td valign=top>Crossref</td><td>".$publication['crossref'];
    
    $Q = mysql_query("SELECT * FROM publication WHERE bibtex_id='".addslashes(trim($publication['crossref']))."'");
    
    if (mysql_num_rows($Q) > 0)
    {
      $R = mysql_fetch_array($Q);
      $result .= ":<br/><div class=message>".getPublicationSummaryByRow($R, "plain", $_SESSION['summarystyle'])."</div>";
    }
      
    $result .= "</td></tr>\n";
  }
  

  //attachments
  if (haveRights("PDF"))
  { 
    $result .= "<tr><td valign=top>Attachments</td><td><table class=vlist cellspacing='0'>";
  
    $Q = mysql_query("SELECT * FROM publicationfile WHERE pub_id =".$pub_id." AND ismain='TRUE'");
    if ($Q)
    {
      if ($R = mysql_fetch_array($Q))
      {
           
        if ($R["note"] && ($R["note"]!="") && ($R["note"] != "main file")) 
        {
          $attachmentDescription = $R['note'];
        }
        else 
        {
          $attachmentDescription = $R['name'];
        }
        
        $result .= "<tr><td>".getAttachmentDownloadLink($R)." ".$attachmentDescription." (main file) ";
        if (haveRights("EDIT"))
        {
          $result .= "</td><td>";
          $result .= getAttachmentEditLink($pub_id, $R['location']);
          $result .= getLinkToAttDeletePage($pub_id, $R['location']);
          $result .= getAttachmentUnsetMainLink($pub_id, $R["location"]);
        }
        $result .= "</td></tr>\n";
      }
    }
    
    $Q = mysql_query("SELECT * FROM publicationfile WHERE pub_id=".$pub_id." AND ismain='FALSE'");
    if ($Q)
    {
      while ($R = mysql_fetch_array($Q))
      {
        $result .= "<tr><td>";
        
        if ($R["note"] && ($R["note"]!="")) 
        {
          $attachmentDescription = $R['note'];
        }
        else 
        {
          $attachmentDescription = $R['name'];
        }
        $result .= getAttachmentDownloadLink($R)." ";
        $result .= $attachmentDescription." ";
        
        if (haveRights("EDIT"))
        {
          $result .= "</td><td>";
          $result .= getAttachmentEditLink($pub_id, $R['location']);
          $result .= getLinkToAttDeletePage($pub_id, $R["location"]);
          $result .= getAttachmentSetMainLink($pub_id, $R["location"]);
        }
        $result .= "</td></tr>\n";
      }
    }
    if (haveRights("EDIT"))
    {
      $result .= "<tr><td colspan=2>".getLinkToAttAddPage($pub_id)."</td></tr>\n";
    }
    $result .= "</table></td></tr>\n";
  }
  
  //mark
  $Q = mysql_query("SELECT mark FROM publication WHERE pub_id=".$pub_id);
  $R = mysql_fetch_array($Q);
  $result .= "<tr><td>Total mark: </td><td>".$R['mark']."</td></tr>\n";
  
  //mark change?
  if (haveRights("COMMENT"))
  {
    $result .= "<tr><td valign=top>Add your mark</td><td>".getMarkAddForm($pub_id)."</td></tr>\n";
  }

  //display notes
  $result .= "<tr><td colspan=2>".getAllNotesForPublicationByPubID($pub_id)."<br/>";
  $result .= getLinkToNoteAddPage($pub_id)."</td></tr>\n";
  
  //and close the encapsulating table environment
  $result .= "</table>";

  return $result;
}

function getPublicationSummaryByID($pub_id, $tableFormat = "inTable", $displayFirst = "author", $checkbox = "false") 
{
  $result = "";
  $Q = mysql_query("SELECT * FROM publication WHERE pub_id=".$pub_id);
  if (mysql_num_rows($Q) != 1)
  {
    $result .= "<div class='errormessage'>An error occurred while fetching the publication with ID ".$pub_id.". ";
    $result .= "Please try again or ask your aigaion administrator.</div>";
    return $result;
  }
  $R = mysql_fetch_array($Q);
  
  $result .= getPublicationSummaryByRow($R, $tableFormat, $displayFirst, $checkbox);

  return $result;
}

function getPublicationSummaryByRow($publicationRow, $tableFormat = "inTable", $displayFirst = "author", $checkbox = "false")
{
  if ($publicationRow['specialchars'] == "TRUE")
  {
    prettyPrintBibCharsFromArray($publicationRow);
  }
  $result  = "";
  $result .= getPublicationSummaryByRowNoNotes($publicationRow, $tableFormat, $displayFirst, $checkbox);
 
  $notes = getAllNotesForPublicationByPubID($publicationRow['pub_id'], $tableFormat);

  if ($notes != "")
  {
    if ($tableFormat == "inTable")
    {
      $result .= "<tr><td>&nbsp;</td><td>";
    }
    $result .= $notes."<br/>";
    if ($tableFormat == "inTable")
    {
      $result .= "</td></tr>\n";
    }  
  }
  
  return $result;
}

function getPublicationSummaryByRowNoNotes($publicationRow, $tableFormat = "inTable", $displayFirst = "author", $checkbox = "false") 
{
	$result = "";
	
	//if crossref is set, we try to fetch the corresponding publication 
	//from the database and merge fields.
	if (trim($publicationRow['crossref']) != "")
	{
	  $Q = mysql_query("SELECT * FROM publication WHERE bibtex_id='".addslashes(trim($publicationRow['crossref']))."'");
    
    if (mysql_num_rows($Q) > 0)
    {
      $R = mysql_fetch_array($Q);
    }
    $keys = array_keys($publicationRow);
    foreach ($keys as $key)
    {
      if ((trim($publicationRow[$key]) == "") && (trim($R[$key]) != ""))
      {  
        $publicationRow[$key] = $R[$key];
      }
    }
	}
	
	$pub_id = $publicationRow['pub_id'];
  
  if ($tableFormat == "inTable")
	{
		
		$result.="<tr><td valign=top><nobr>";
	}

  //CHECKBOX FOR MULTI-PUB ACTIONS
  if ($checkbox == "true")
  {
    $result .= getPublicationCheckForm($pub_id)."&nbsp;";
  }
                  
  //BIBTEX link
  $result .= "<span alt='Get BibTeX for ".$publicationRow["bibtex_id"]."' title='Get BibTeX for ".$publicationRow["bibtex_id"]."'>".getBibtexLinkByID($pub_id)."</span>&nbsp;";
  //RIS link
  $result .= "<span alt='Get RIS for ".$publicationRow["bibtex_id"]."' title='Get RIS for ".$publicationRow["bibtex_id"]."'>".getRISLinkByID($pub_id)."</span>&nbsp;";
  
  
  //edit publication link, if user has sufficient rights
  if (haveRights("EDIT")) 
  {
      $result .= getLinkToPublicationEditPage($pub_id, "small")."";
  }

  //view publication file link, if user has sufficient rights
  if (haveRights("PDF"))
  {
    $Q = mysql_query("SELECT * FROM publicationfile WHERE pub_id =".$pub_id." AND ismain='TRUE';");
    if ($Q) 
    {
      if ($R = mysql_fetch_array($Q))
      {
          $result .= getAttachmentDownloadLink($R);
      }
    }
  }
  
  if ($tableFormat == "inTable")
	{
		$result.="</nobr></td><td>";
	}
    
  if ($displayFirst == "author")
  {
    $result .= getAuthorsForPublication($pub_id)." ";
    $result .= getLinkToPublicationDisplayPage($pub_id, $publicationRow['title']).", ";
  }
  else
  {
  	$result .= getLinkToPublicationDisplayPage($pub_id, $publicationRow['title']).", ";
  	$result .= getAuthorsForPublication($pub_id)." ";
  }
  $result .= getTypeSpecificElementsSummary($publicationRow, "plain");
  if ($tableFormat == "inTable")
  {
    $result .= "</td></tr>\n";
  }
  return $result;
}

function getLinkToPublicationDisplayPage($pub_id, $title)
{
  global $URL, $THEME;
  quotesToHTMLFromString($title);
  $result  = "";
  $result .= "<i><a href=".$ROOT."?page=publication&kind=single&ID=".$pub_id.">".$title."</a></i>";
  return $result;
}

function getLinkToPublicationEditPage($pub_id, $size = "normal") {
  global $URL,$THEME;
  $return=rawurlencode($_SERVER['PHP_SELF']."?".$_SERVER["QUERY_STRING"]);  
  $result  = "";
  $result .= '<a class=small href="index.php?page=publication&kind=edit&ID='.$pub_id.'&return='.$return.'">';
  
  if ($size == "normal")
  {
    $result .= "<IMG SRC='".$URL.$THEME."icons/_edit.gif' border=0>";
  }
  else if ($size == "small")
  {
    $result .= "<IMG SRC='".$URL.$THEME."icons/_edit_small.gif' border=0>";
  }
  
  $result .= "</a>";

  return $result;
}

function getLinkToPublicationDeletePage($pub_id, $size = "normal", $return="") {
  global $URL,$THEME;
  if ($return=="") {
    $return=$_SERVER['PHP_SELF']."?".$_SERVER["QUERY_STRING"];
  }
  
  $result  = "";
  $result .= '<a class=small href="index.php?page=publication&kind=delete&ID='.$pub_id.'&return='.rawurlencode($return).'">';
  
  if ($size == "normal")
  {
    $result .= "<IMG SRC='".$URL.$THEME."icons/_delete.gif' border=0>";
  }
  else if ($size == "small")
  {
    $result .= "<IMG SRC='".$URL.$THEME."icons/_delete_small.gif' border=0>";
  }
  
  $result .= "</a>";

  return $result;
}

function getTypeSpecificElements($publicationRow) {
  
  $elements = bibGetFieldArray($publicationRow['type']);
  $pageNrs = "";
  if (in_array("pages", $elements)) {
    if ($publicationRow['firstpage']) {
      $pageNrs = $publicationRow['firstpage']."-".$publicationRow['lastpage'];
    }
  }
  if (in_array("year", $elements)) {
  	if ($publicationRow["year"] == "0000") {
  		$publicationRow["year"] = "Unknown";
  	}
  }
  $abstract = "";
  if (in_array("abstract", $elements))
  {
    if ($publicationRow['abstract'])
    {
      $abstract = "<DIV class='readernote'>";
      $abstract .= $publicationRow['abstract'];
      $abstract .= "</DIV>";
    }
  }
  $month = "";
  if (in_array("month", $elements)) {
    global $MONTHS_ENG;
    $month = $MONTHS_ENG[$publicationRow['month']];
  }
  $url = "";
  if (in_array("url", $elements))
  {
    if ($publicationRow['url']) {
      $url = html_activate_links($publicationRow['url']);
    }
  }
  
  $keywords = "";
  if (in_array("keywords", $elements))
  {
   if ($publicationRow['keywords']) 
    {
        //echo "bla";
        //split keywords, make search links, add them together with comments..
        //return  $publicationRow['keywords'];
        $keywords = preg_replace('/ *([^,]+)/',"<A HREF=\"".$DIR."index.php?page=search&command=search&keywords=\\1\">\\1</A>",$publicationRow['keywords']);
    } 
  }
  
  
  $result .= getElement($elements, "journal",      $publicationRow['journal'],      "Journal");
  $result .= getElement($elements, "booktitle",    $publicationRow['booktitle'],    "Booktitle");
  $result .= getElement($elements, "series",       $publicationRow['series'],       "Series");
  $result .= getElement($elements, "year",         $publicationRow['year'],         "Year published");
  $result .= getElement($elements, "month",        $month,                          "Month");
  $result .= getElement($elements, "volume",       $publicationRow['volume'],       "Volume");
  $result .= getElement($elements, "number",       $publicationRow['number'],       "Number");
  $result .= getElement($elements, "edition",      $publicationRow['edition'],      "Edition");
  $result .= getElement($elements, "pages",        $pageNrs,                          "Pages");
  $result .= getElement($elements, "chapter",      $publicationRow['chapter'],      "Chapter");
  $result .= getElement($elements, "publisher",    $publicationRow['publisher'],    "Publisher");
  $result .= getElement($elements, "school",       $publicationRow['school'],       "School");
  $result .= getElement($elements, "organization", $publicationRow['organization'], "Organization");
  $result .= getElement($elements, "institution",  $publicationRow['institution'],  "Institution");
  $result .= getElement($elements, "location",     $publicationRow['location'],     "Location");
  $result .= getElement($elements, "address",      $publicationRow['address'],      "Address");
  $result .= getElement($elements, "howpublished", $publicationRow['howpublished'], "How published");
  $result .= getElement($elements, "isbn",         $publicationRow['isbn'],         "ISBN");
  $result .= getElement($elements, "issn",         $publicationRow['issn'],         "ISSN");
  $result .= getElement($elements, "note",         $publicationRow['note'],         "Note");
  $result .= getElement($elements, "url",          $url,                            "URL");
  $result .= getElement($elements, "keywords",     $keywords,                       "Keywords");
  $result .= getElement($elements, "abstract",     $abstract,                       "Abstract");
  
  return $result;
}

function getTypeSpecificElementsSummary($publicationRow) {
  
  $elements = bibGetSummaryFieldArray($publicationRow['type']);
  if (in_array("pages", $elements)) {
    $pageNrs = "";
    if ($publicationRow['firstpage']) {
      $pageNrs = $publicationRow['firstpage']."-".$publicationRow['lastpage'];
    }
  }
  
  if (in_array("year", $elements)) {
  	if ($publicationRow["year"] == "0000") {
  		$publicationRow["year"] = "";
  	}
  }
  
  if (in_array("month", $elements)) {
    global $MONTHS_ENG;
    $month = $MONTHS_ENG[$publicationRow['month']];
  }
  
  $result .= getElement($elements, "journal",      $publicationRow['journal'],      "in: ",     "plain");
  $result .= getElement($elements, "booktitle",    $publicationRow['booktitle'],    "in: ",     "plain");
  $result .= getElement($elements, "series",       $publicationRow['series'],       "",         "plain");
  $result .= getElement($elements, "volume",       $publicationRow['volume'],       "volume ",  "plain");
  $result .= getElement($elements, "number",       $publicationRow['number'],       "number ",  "plain");
  $result .= getElement($elements, "pages",        $pageNrs,                          "pages ",   "plain");
  $result .= getElement($elements, "chapter",      $publicationRow['chapter'],      "chapter ", "plain");
  $result .= getElement($elements, "publisher",    $publicationRow['publisher'],    "",         "plain");
  $result .= getElement($elements, "school",       $publicationRow['school'],       "",         "plain");
  $result .= getElement($elements, "organization", $publicationRow['organization'], "",         "plain");
  $result .= getElement($elements, "location",     $publicationRow['location'],     "",         "plain");
  $result .= getElement($elements, "howpublished", $publicationRow['howpublished'], "",         "plain");
  $result .= getElement($elements, "isbn",         $publicationRow['isbn'],         "ISBN " ,   "plain");
  $result .= getElement($elements, "issn",         $publicationRow['issn'],         "ISSN " ,   "plain");
  $result .= getElement($elements, "month",        $month,                          "",         "plain");
  $result .= getElement($elements, "year",         $publicationRow['year'],         "",         "plain");
  
  $result = rtrim($result, ", ");
  return $result.".";
}

function getElement($elements, $element, $data, $name, $tableFormat = "inTable")
{
  $result = "";
  if (in_array($element, $elements) && trim($data) != "") {
    if ($tableFormat == "inTable")
    {
      $result .= "<tr><td valign=top>".$name." </td><td>".$data."</td></tr>\n";
    }
    else
    {
      $result .= $name.$data.", ";
    }
  }
  return $result;
}